﻿/***********************************************************************************************************************
/*
/*	cSunriseEquation.j.cpp
/*
/*	Begun January 27, 2019
/*
/*	Copyright Robert Sacks.  https://mojoware.org
/*
/**********************************************************************************************************************/

// #include "stdafx.h"
#include <Windows.h>
#include "cSunriseEquation.j.h"
#include <cmath>
#include <assert.h>
#include <stdint.h>

// #define PURE_EPOCH

//======================================================================================================================
//  PROTOTYPES
//======================================================================================================================

//======================================================================================================================
//  DATA
//======================================================================================================================

static const UINT64 uUnitsInMillisecond		= 10000;
static const UINT64 uUnitsInSecond			= 1000 * uUnitsInMillisecond; // 10000000;
static const UINT64 uUnitsInMinute			= 60 * uUnitsInSecond;
static const UINT64 uUnitsInHour			= 60 * uUnitsInMinute;
static const UINT64 uUnitsInDay				= 24 * uUnitsInHour;

static const time_t ttUnix2000 = 946728000; // 2000-jan-1 12:00 UTC
static const double dPi = 3.14159265359;
static const double dJulian2000 = 2451545.0; // Julian date 2000-jan-1 12:00 UTC
static const unsigned uSecondsInDay = 24 * 60 * 60;

//======================================================================================================================
//  CODE
//======================================================================================================================

//----------------------------------------------------------------------------------------------------------------------
//  JULIAN TO LOCAL
//----------------------------------------------------------------------------------------------------------------------
bool cSunriseEquation :: julian_to_local ( time_t * pRetUnix, double j )
{
	TIME_ZONE_INFORMATION tzi = {0};

	DWORD dwResult = GetTimeZoneInformation (&tzi);

	if ( TIME_ZONE_ID_INVALID == dwResult )
	{
		assert(0);
		return false;
	}

	double d = j - dJulian2000;

	time_t u  = ttUnix2000;

	u += int ( d * 24.0 * 60.0 * 60.0 );
	u -=  int ( tzi.Bias * 60.0 ); // adjust for time zone

	*pRetUnix = u;

	return true;
}


//----------------------------------------------------------------------------------------------------------------------
//  JULIAN TO LOCAL
//----------------------------------------------------------------------------------------------------------------------
bool cSunriseEquation :: julian_to_utc ( cTime2 * pRet, double j )
{
	double d = j - dJulian2000;

	cTime2 t ( 2000, 1, 1, 12 ); // julian days start at noon
	uint64_t u = static_cast<uint64_t>(t);
	u += uint64_t ( d * 24.0 * 60.0 * 60.0 * 10000000.0 );
	cTime2 ret ( u );

	*pRet = ret;

	return true;
}

//----------------------------------------------------------------------------------------------------------------------
//  JULIAN TO LOCAL
//----------------------------------------------------------------------------------------------------------------------
#if 0
bool cSunriseEquation :: julian_to_local ( cTime2 * pRet, double j )
{
	TIME_ZONE_INFORMATION tzi = {0};

	DWORD dwResult = GetTimeZoneInformation (&tzi);

	if ( TIME_ZONE_ID_INVALID == dwResult )
	{
		assert(0);
		return false;
	}

	double d = j - dJulian2000;

	cTime2 t ( 2000, 1, 1, 12 ); // julian days start at noon
	UINT64 u = t;
	u += UINT64 ( d * 24.0 * 60.0 * 60.0 * 10000000 );
	u -=  UINT64 ( tzi.Bias * 60.0 * 10000000 ); // adjust for time zone
	cTime2 ret ( u );
	
	*pRet = ret;

	return true;
}
#endif

//----------------------------------------------------------------------------------------------------------------------
//  DEGREES TO RADIANS
//----------------------------------------------------------------------------------------------------------------------
double cSunriseEquation :: deg_to_rad ( double dDeg )
{
	return dDeg * dPi / 180.0;
}


//----------------------------------------------------------------------------------------------------------------------
//  RADIANS TO DEGREES
//----------------------------------------------------------------------------------------------------------------------
double cSunriseEquation :: rad_to_deg ( double dRad )
{
	return dRad * 180.0 / dPi;
}


//----------------------------------------------------------------------------------------------------------------------
//  DO IT USING UNIX TIME
//  Implements algorithm on Wikipedia page "Sunrise Equation".
//----------------------------------------------------------------------------------------------------------------------
#if 0
bool cSunriseEquation :: do_it_unix ( time_t * pRise, time_t * pSet, double dLatitudeDeg, double dLongitudeDeg, time_t ttQuery )
{
	double dLongitude = deg_to_rad ( dLongitudeDeg );
	double dLatitude = deg_to_rad ( dLatitudeDeg );

	// CALCULATE NUMBER OF JULIAN DAYS SINCE 2000-JAN-1 12:00 GMT

	double dN;
	{
		assert ( ttUnix2000 < ttQuery );
		// set query to noon on that day
		ttQuery = ( ttQuery / uSecondsInDay ) * uSecondsInDay + uSecondsInDay / 2;
		assert ( ttUnix2000 < ttQuery );
		double dSec = (double) ( ttQuery - ttUnix2000 );
		dN = dSec / uSecondsInDay;
		dN += 0.0008;
	}

	// MEAN SOLAR NOON

	double dJStar = dN - ( dLongitude / ( 2 * dPi ) );

	// SOLAR MEAN ANOMALY

	double dM_deg = fmod ( 357.5291 + 0.98560028 * dJStar, 360.0 );
	double dM = deg_to_rad ( dM_deg );

	// EQUATION OF THE CENTER

	double dC = 1.9148 * sin(dM) + 0.0200 * sin(2*dM) + 0.0003 * sin ( 3*dM);

	// ECLIPTIC LONGITUDE

	double dLambda = deg_to_rad ( fmod ( dM_deg + dC + 180.0 + 102.9372, 360.0 ) );

	// SOLAR JTRANSIT

	double dJTransit = dJulian2000 + dJStar + 0.0053 * sin ( dM ) - 0.0069 * sin ( 2 * dLambda );

	// DECLINATION OF THE SUN

	double dDelta = asin ( sin ( dLambda ) * sin ( deg_to_rad ( 23.44 ) ) );

	// HOUR ANGLE ω

	double dCosOmega = (sin(deg_to_rad(-0.83)) - sin(dLatitude) * sin(dDelta) ) 
		/ ( cos ( dLatitude ) * cos(dDelta) );

	double dOmega = acos ( dCosOmega );

	double dSunrise = dJTransit -  dOmega / ( 2 * dPi);
	double dSunset = dJTransit +  dOmega / ( 2 * dPi);

	julian_to_local ( pRise, dSunrise );
	julian_to_local ( pSet, dSunset );

	return true;
}
#endif

//----------------------------------------------------------------------------------------------------------------------
//  DO IT USING cTime2 I.E. WINDOWS FILETIME
//  Implements algorithm on Wikipedia page "Sunrise Equation".
//  Input:  tQuery is a date; time will be discarded (tQuery will be treated as noon UTC on that date)
//  Output: Local times of sunrise and sunset at specified latitude/longitude on tQuery UTC date.
//----------------------------------------------------------------------------------------------------------------------
bool cSunriseEquation :: do_it ( bool * pAlwaysDay, bool * pAlwaysNight, cTime2 * pRiseUtc, cTime2 * pSetUtc, double dLatitudeDeg, double dLongitudeDeg, cTime2 tQueryUtcA )
{
	if ( pAlwaysDay )
		*pAlwaysDay = false;
	
	if ( pAlwaysNight )
		*pAlwaysNight = false;

	if ( ! ( ( -90.0 <= dLatitudeDeg ) && ( dLatitudeDeg <= 90.0 ) ) )
		return false;

	if ( ! ( ( -180.0 <= dLongitudeDeg ) && ( dLongitudeDeg <= 180.0 ) ) )
		return false;

	double dLongitude = deg_to_rad ( dLongitudeDeg );
	double dLatitude  = deg_to_rad ( dLatitudeDeg );

	// CALCULATE NUMBER OF JULIAN DAYS SINCE 2000-JAN-1 12:00 GMT

	double dN;
	{
		cTime2 tEpoch ( 2000, 1, 1, 12 );
		cTime2 tQueryUtc ( tQueryUtcA.date_only() + 12 * nTime2::HOUR );
		double dSec = double ( ( tQueryUtc - tEpoch ) / nTime2::SEC );
		dN = dSec / ( 24 * 60 * 60 );
		dN += 0.0008;
	}

	// MEAN SOLAR NOON

	double dJStar = dN - ( dLongitude / ( 2 * dPi ) );

	// SOLAR MEAN ANOMALY

	double dM_deg = fmod ( 357.5291 + 0.98560028 * dJStar, 360.0 );
	double dM = deg_to_rad ( dM_deg );

	// EQUATION OF THE CENTER

	double dC = 1.9148 * sin(dM) + 0.0200 * sin(2*dM) + 0.0003 * sin ( 3*dM);

	// ECLIPTIC LONGITUDE

	double dLambda = deg_to_rad ( fmod ( dM_deg + dC + 180.0 + 102.9372, 360.0 ) );
 
	// SOLAR JTRANSIT

	double dJTransit = dJulian2000 + dJStar + 0.0053 * sin ( dM ) - 0.0069 * sin ( 2 * dLambda );

	// DECLINATION OF THE SUN

	double dDelta = asin ( sin ( dLambda ) * sin ( deg_to_rad ( 23.44 ) ) );

	// HOUR ANGLE ω

	double dCosOmega = (sin(deg_to_rad(-0.83)) - sin(dLatitude) * sin(dDelta) ) 
		/ ( cos ( dLatitude ) * cos(dDelta) );

	//-----------------------------------
	//  RANGE CHECKING ADDED 2019 MAR 22
	//-----------------------------------

	if ( dCosOmega <= -1.0 )
	{
		// always light
		if ( pAlwaysDay )
			*pAlwaysDay = true;
		return true;
	}

	else if ( 1.0 <= dCosOmega )
	{
		// always dark
		if ( pAlwaysNight )
			*pAlwaysNight = true;
		return true;
	}

	double dOmega = acos ( dCosOmega );

	// 0 == dOmega; always dark
	// dPi == dOmega; always light

	double dSunrise = dJTransit -  dOmega / ( 2 * dPi);
	double dSunset = dJTransit +  dOmega / ( 2 * dPi);

	julian_to_utc ( pRiseUtc, dSunrise );
	julian_to_utc ( pSetUtc,  dSunset );

	return true;
}
