/***********************************************************************************************************************
/*
/*  cCrypt_rotate.j.h
/*
/*  For obfuscation only of blobs, strings, and integers.
/*	Super fast.
/*
/*	Copyright 2020-21 Robert Sacks.  https://mojoware.org
/*
/**********************************************************************************************************************/

#pragma once

#include <stdint.h>   // for uint64_t
#include <limits.h>   // for CHAR_BIT

class cCrypt_rotate
{
public:

	static void encrypt ( void * pa, size_t size_in_bytes, 
		unsigned uRot = _uRot, uint64_t uXor = _uXor, uint64_t uMult = _uMult );

	static void decrypt ( void * pa, size_t size_in_bytes, 
		unsigned uRot = _uRot, uint64_t uXor = _uXor, uint64_t uMult = _uMult );

	template<typename T> static T encrypt_int ( T u, unsigned uRot = _uRot, uint64_t uXor = _uXor );
	template<typename T> static T decrypt_int ( T u, unsigned uRot = _uRot, uint64_t uXor = _uXor );

private:

	static const unsigned _uRot  = 31;
	static const uint64_t _uXor  = 0x7e327bc904da8321;
	static const uint64_t _uMult = 0x456789abba987654;

	template<typename T> static T rotl ( T n, unsigned int c );
	template<typename T> static T rotr ( T n, unsigned int c );
};

#pragma warning ( push )
#pragma warning ( disable : 4146 ) // unary minus on uint

//----------------------------------------------------------------------------------------------------------------------
//  ROTATE LEFT
//----------------------------------------------------------------------------------------------------------------------
template<typename T>
T cCrypt_rotate :: rotl ( T n, unsigned int c )
{
	const unsigned int mask = ( CHAR_BIT*sizeof(T) - 1 );
	c &= mask;
	return (n<<c) | (n>>( (-c)&mask ));
}


//----------------------------------------------------------------------------------------------------------------------
//  ROTATE RIGHT
//----------------------------------------------------------------------------------------------------------------------
template<typename T>
T cCrypt_rotate :: rotr ( T n, unsigned int c )
{
	const unsigned int mask = ( CHAR_BIT*sizeof(n) - 1 );
	c &= mask;
	return (n>>c) | (n<<( (-c)&mask ));
}

#pragma warning ( pop )

//----------------------------------------------------------------------------------------------------------------------
//  ENCRYPT UNSIGNED INT
//----------------------------------------------------------------------------------------------------------------------
template<typename T> 
T cCrypt_rotate :: encrypt_int ( T u, unsigned uRot, uint64_t uXor )
{
	u = rotl ( u, uRot );
	u ^= (T) uXor;
	return u;
}


//----------------------------------------------------------------------------------------------------------------------
//  DECRYPT UNSIGNED INT
//----------------------------------------------------------------------------------------------------------------------
template <typename T>
T cCrypt_rotate :: decrypt_int ( T u, unsigned uRot, uint64_t uXor )
{
	u ^= (T) uXor;
	u = rotr ( u, uRot );
	return u;
}

